-- Common symbology definitions. Used by HMD

dofile(LockOn_Options.common_script_path.."elements_defs.lua")
dofile(LockOn_Options.script_path.."materials.lua")

local use_mipfilter 	= true
local show_masks  		= false

-- made by four lines
line_box_indices  		= {0, 1, 1, 2, 2, 3, 3, 0}

-- overwrite this value with actual clipping level
DEFAULT_LEVEL     		= 0
-- fill this table with font parameters
stringdefs 				= {}
-- overwrite these values with actual material names
default_material 		= ""
stroke_font				= ""

function addPlaceholder(name, pos, parent, controllers)
	local placeholder       	= CreateElement "ceSimple"
	placeholder.name        	= name
	pos = pos or {0, 0}
	placeholder.init_pos    	= {pos[1], pos[2], 0}
	placeholder.collimated  	= collimated or false

	if parent ~= nil then
		placeholder.parent_element 	= parent
	end	
	
	if controllers ~= nil then
		placeholder.controllers		= controllers
	end
	
	Add(placeholder)
	return placeholder
end

function setAsInvisibleMask(obj)
	obj.isvisible = show_masks
	obj.material  = "MASK_MATERIAL_PURPLE"
end

function setClipLevel(obj, level)
	level = level or 0
	obj.h_clip_relation = h_clip_relations.COMPARE
	obj.level 		    = DEFAULT_LEVEL + level
end

-- Local functions

-- NOTE
-- 'pos' is passed as a two-component table - x and y coordinates
function setSymbolCommonProperties(symbol, name, pos, parent, controllers, material)
	symbol.name               = name
	symbol.isdraw             = true
	symbol.material           = material or default_material
	symbol.additive_alpha	  = additive_alpha or false
	symbol.collimated		  = collimated or false
	symbol.use_mipfilter      = use_mipfilter
	
	if parent ~= nil then
		symbol.parent_element = parent
	end
	
	if controllers ~= nil then
		if type(controllers) == "table" then
			symbol.controllers = controllers
		end
	end
	
	pos = pos or {0, 0}
	symbol.init_pos       	  = {pos[1], pos[2], 0}
	
	setClipLevel(symbol)
end

local function setSymbolAlignment(symbol, align)
	if align ~= nil then
		symbol.alignment = align
	else
		symbol.alignment = "CenterCenter"
	end
end

local function setStrokeSymbolProperties(symbol)

	if override_materials == true then
		-- Is used for outlined font generated by DMC
		symbol.thickness    		= override_thickness
		symbol.fuzziness    		= override_fuzziness
	else
		symbol.thickness    		= 0.8
		symbol.fuzziness    		= 0.5
	end

	if JHMCSII == true then
		symbol.thickness = 0.6
		symbol.fuzziness = 0.4
	end

	symbol.draw_as_wire 		= dbg_drawStrokesAsWire
	--symbol.use_specular_pass 	= false -- ommitted for now as is set for the entire indicator
end

local function buildStrokeLineVerts(length, dashed, stroke, gap)
	local verts = {}
	local inds = {}
	
	if dashed == true and stroke ~= nil and gap ~= nil then
		local segLength = stroke + gap
		local numOfWholePairs = math.floor(length / segLength)
		local reminder = length - numOfWholePairs * segLength
		
		local function addSeg(num)
			local shift1 = num * 2
			verts[shift1 + 1] = {0, num * segLength}
			verts[shift1 + 2] = {0, num * segLength + stroke}
			
			inds[shift1 + 1] = shift1
			inds[shift1 + 2] = shift1 + 1
		end
		
		for segNum = 0, numOfWholePairs - 1 do
			addSeg(segNum)
		end
		
		if reminder > 0 then
			if reminder >= stroke then
				addSeg(numOfWholePairs)
			else
				local shift1 = numOfWholePairs * 2
				verts[shift1 + 1] = {0, numOfWholePairs * segLength}
				verts[shift1 + 2] = {0, numOfWholePairs * segLength + reminder}
				
				inds[shift1 + 1] = shift1
				inds[shift1 + 2] = shift1 + 1
			end
		end
	else
		verts = {{0, 0}, {0, length}}
		inds  = {0, 1}
	end
	
	return verts, inds
end

-- Shared functions

-- Stroke text with glyphs described in a .svg file
function addStrokeText(name, value, stringdef, align, pos, parent, controllers, formats)
	local txt = CreateElement "ceStringSLine"
	setSymbolCommonProperties(txt, name, pos, parent, controllers, stroke_font)
	setSymbolAlignment(txt, align)
	
	-- custom size is noted in documents as in percents from the original one
	if type(stringdef) == "table" then
		txt.stringdefs = stringdef
	else
		txt.stringdefs = stringdefs[stringdef]
	end
	
	if value ~= nil then
		txt.value = value
	end
	
	txt.formats 		= formats
	
	Add(txt)
		
	return txt
end

-- Stroke symbol with points described in a .svg file
function addStrokeSymbol(name, set, align, pos, parent, controllers, scale, material)
	local symbol       = CreateElement "ceSMultiLine"
	setSymbolCommonProperties(symbol, name, pos, parent, controllers, material)
	setSymbolAlignment(symbol, align)
	setStrokeSymbolProperties(symbol)
	symbol.points_set  = set
	symbol.scale       = scale or 1
	Add(symbol)
	return symbol
end

-- Stroke line
-- rot (CCW in degrees from up)
-- pos (position of beginning of the line)
function addStrokeLine(name, length, pos, rot, parent, controllers, dashed, stroke, gap, material)
	local line      = CreateElement "ceSMultiLine"
	setSymbolCommonProperties(line, name, pos, parent, controllers, material)
	setStrokeSymbolProperties(line)
	
	if rot ~= nil then
		line.init_rot   = {rot}
	end
		
	local verts, inds = buildStrokeLineVerts(length, dashed, stroke, gap)
	line.vertices   = verts
	line.indices    = inds
	
	Add(line)
	return line
end

function addStrokeLineLeveled(name, length, pos, rot, parent, controllers)
	local line      = CreateElement "ceSMultiLine"
	setSymbolCommonProperties(line, name, pos, parent, controllers, material)
	setStrokeSymbolProperties(line)
	
	if rot ~= nil then
		line.init_rot   = {rot}
	end
		
	local verts, inds = buildStrokeLineVerts(length, dashed, stroke, gap)
	line.vertices   = verts
	line.indices    = inds
	
	line.h_clip_relation = h_clip_relations.REWRITE_LEVEL
	line.level 		    = DEFAULT_LEVEL

	Add(line)
	return line
end

-- Stroke line of variable length
function addVarLenStrokeLine(name, length, pos, rot, parent, controllers, dashed, stroke, gap, material)
	local line      = CreateElement "ceSVarLenLine"
	setSymbolCommonProperties(line, name, pos, parent, controllers, material)
	setStrokeSymbolProperties(line)
	
	if rot ~= nil then
		line.init_rot   = {rot}
	end
	
	if length ~= nil then
		line.length     = length
	end
	
	local verts, inds = buildStrokeLineVerts(HUD_TFOV_DI * 4, dashed or false, stroke, gap)
	line.vertices   = verts
	line.indices    = inds
	
	Add(line)
	return line
end

-- Box made of stroke lines. Use case - HUD TD box (non-segmented)
function addStrokeBox(name, sideX, sideY, align, pos, parent, controllers, material)
	local box      = CreateElement "ceSMultiLine"
	setSymbolCommonProperties(box, name, pos, parent, controllers, material)
	setSymbolAlignment(box, align)
	setStrokeSymbolProperties(box)
	
	local halfSideX = sideX / 2
	local halfSideY = sideY / 2
	box.vertices    = {{-halfSideX, -halfSideY}, {-halfSideX, halfSideY}, {halfSideX, halfSideY}, {halfSideX, -halfSideY}}
	box.indices     = line_box_indices
	
	Add(box)
	return box
end

-- Segmented box made of stroke lines. Use case - HUD segmented TD box
function addStrokeBoxSegmented(name, sideX, sideY, segmentX, segmentY, align, pos, parent, controllers, material)
	local box      = CreateElement "ceSMultiLine"
	setSymbolCommonProperties(box, name, pos, parent, controllers, material)
	setSymbolAlignment(box, align)
	setStrokeSymbolProperties(box)
	
	local halfSideX = sideX / 2
	local halfSideY = sideY / 2
	local lineLenX  = halfSideX - segmentX / 2
	local lineLenY  = halfSideY - segmentY / 2
	box.vertices    = {{-halfSideX, -halfSideY}, {-halfSideX, -halfSideY + lineLenY}, {-halfSideX, halfSideY - lineLenY},
					   {-halfSideX, halfSideY}, {-halfSideX + lineLenX, halfSideY}, {halfSideX - lineLenX, halfSideY},
					   {halfSideX, halfSideY}, {halfSideX, halfSideY - lineLenY}, {halfSideX, -halfSideY + lineLenY},
					   {halfSideX, -halfSideY}, {halfSideX - lineLenX, -halfSideY}, {-halfSideX + lineLenX, -halfSideY}}
	box.indices     = {0, 1, 2, 3, 3, 4, 5, 6, 6, 7, 8, 9, 9, 10, 11, 0}
	
	Add(box)
	return box
end

-- Equilateral triangle. Center point is at the top end.
-- The element is rotated about its center point by 'rot' angle, and then 'pos_shift' is applied.
function addStrokeEquilateralTriangle(name, height, angle, pos, rot, pos_shift, parent, controllers, material)
	local triangle      = CreateElement "ceSMultiLine"
	setSymbolCommonProperties(triangle, name, pos, parent, controllers, material)
	setStrokeSymbolProperties(triangle)
	
	local rot_lcl = rot or 0
	-- The angle is limited in the range of 1 to 179 degrees
	local base_angle = 180 - math.max(1, math.min(angle, 179)) / 2
	
	local angle1 = math.rad(-base_angle + rot_lcl)
	local angle2 = math.rad(base_angle + rot_lcl)
	
	local pt2_x = height * math.sin(angle1)
	local pt2_y = height * math.cos(angle1)
	local pt3_x = height * math.sin(angle2)
	local pt3_y = height * math.cos(angle2)

	local shift_x = 0
	local shift_y = 0
	
	if pos_shift ~= nil then
		shift_x = pos_shift[1]
		shift_y = pos_shift[2]
	end
	
	triangle.vertices    = {{shift_x, shift_y}, {pt2_x + shift_x, pt2_y + shift_y}, {pt3_x + shift_x, pt3_y + shift_y}}
	triangle.indices     = {0, 1, 1, 2, 2, 0}
	
	Add(triangle)
	return triangle
end

-- Stroke circle
function addStrokeCircle(name, radius, pos, parent, controllers, arc, segment, gap, dashed, material)
	local segmentsN = 64
	
	local circle     		= CreateElement "ceSCircle"
	setSymbolCommonProperties(circle, name, pos, parent, controllers, material)
	setStrokeSymbolProperties(circle)
	circle.radius    	 	= {radius, radius}
	circle.arc       		= arc or {0, math.pi * 2}
	circle.segment      	= segment or math.pi * 4 / segmentsN
	circle.gap      		= gap or math.pi * 4 / segmentsN
	circle.segment_detail   = 4
	
	if dashed ~= nil then
		circle.dashed    	= dashed
	else
		circle.dashed    	= false
	end
	
	Add(circle)
	return circle
end

-- Non-textured mesh, which is drawn by either triangles or by zero-width lines ('primitives' parameter)
function addMesh(name, vertices, indices, pos, primitives, parent, controllers, material)
	local mesh     			= CreateElement "ceMeshPoly"
	setSymbolCommonProperties(mesh, name, pos, parent, controllers, material)
	mesh.vertices			= vertices
	mesh.indices			= indices
	mesh.primitivetype		= primitives
	Add(mesh)
	return mesh
end

-- A symbology clipping region. May be used to limit the area where an element is visible/hidden.
--  In aircraft manuals also called 'occultation area/mask'.
local function createMask(name, vertices, indices, pos, parent, controllers, material)
	local mask            = addMesh(name, vertices, indices, pos, "triangles", parent, controllers)
	setAsInvisibleMask(mask) -- changes material
	mask.additive_alpha	  = false
	mask.change_opacity   = false
	
	if material ~= nil then
		mask.material     = material
	end

	return mask
end

-- See above. Creates the region, and sets its occultation level.
-- Will increase the currently used level value, and will affect symbology 
--  according to 'h_clip_relation' field value of each element rendered after it.
function openMaskArea(level, name, vertices, indices, pos, parent, controllers, material)
	local mask            = createMask(name, vertices, indices, pos, parent, controllers, material)
	mask.h_clip_relation  = h_clip_relations.INCREASE_IF_LEVEL
	mask.level 		      = DEFAULT_LEVEL + level
	return mask
end

-- Same as above, but removes the previously created clipping region.
function closeMaskArea(level, name, vertices, indices, pos, parent, controllers, material)
	local mask            = createMask(name, vertices, indices, pos, parent, controllers, material)
	mask.h_clip_relation  = h_clip_relations.DECREASE_IF_LEVEL
	mask.level 		      = DEFAULT_LEVEL + level
	return mask
end

-- Debug reference grid. Fits for both HUD and MDI/AMPCD
function dbg_add_MDG_RefGrid(step, halfWidth, collimated, noTFOV)
	local lines_count = math.ceil(halfWidth / step)
	if noTFOV == true then
		drawIndicatorRefGrid(lines_count, step, 2 * halfWidth, collimated)
	else
		drawIndicatorRefGrid(lines_count, step, 2 * halfWidth, collimated, halfWidth)
	end
end

-- Debug indicator display center (optical center in the case of HUD)
function dbg_addIndicatorCenter()
	
	do return end -- comment if you want the debug cross to be displayed
	
	-- new way - cross
	local crossSz 		= 40
	local side 			= crossSz * 0.5
	local vertices 		= {{-side, 0}, {side, 0}, {0, side}, {0, -side}}
	local indices  		= {0, 1, 2, 3}
	addMesh("indicator_center", vertices, indices, {0, 0}, "lines", nil, nil, "DBG_RED")
	
	do return end
	
	-- old way - circle
	local dbgCenter         = CreateElement "ceMeshPoly"
	dbgCenter.name          = "indicator_center"
	dbgCenter.primitivetype = "triangles"
	set_circle(dbgCenter, 6, 3)
	dbgCenter.material      = "DBG_RED"
	dbgCenter.collimated    = collimated or false
	Add(dbgCenter)
end

function set_box(obj, left, right, up, down)
	local verts    = {
	{left,  down},
	{left,  up},
	{right, up},
	{right, down}}
	
	local inds = {0, 1, 2,
				  0, 2, 3}
	
	obj.vertices = verts              
	obj.indices  = inds
end

function set_box_w_h(obj, width, height)
	set_box(obj, -width/2, width/2, height/2, -height/2)
end

-- arc is decreased counterclockwise by controllers in the range of 360 - 0 degrees
function addStrokeVarArc(name, radius, rot, pos, parent, controllerMask, controllerMain, level)
	local maskSide = radius
	
	-- right half of the circle (arc)
	local openingMaskRight = openMaskArea(level, name.."_openingMaskRight", {}, {}, pos, parent, controllerMask)
	set_box(openingMaskRight, 0, maskSide + 5, maskSide + 5, -(maskSide + 5))
	
	rot = rot or 0
	
	openingMaskRight.init_rot = {rot - 180}
	--openingMaskRight.isvisible = true -- dbg
	
	addStrokeArc(name.."_right", radius, 180, rot, pos, parent, nil, 1)

	local closingMaskRight = closeMaskArea(level + 1, name.."_closingMaskRight",
		openingMaskRight.vertices, openingMaskRight.indices, openingMaskRight.init_pos, parent, controllerMask)
	closingMaskRight.init_rot = {rot - 180}
	--closingMaskRight.isvisible = true -- dbg
	
	-- left half of the circle (arc)
	local openingMaskLeft = openMaskArea(level, name.."_openingMaskLeft", {}, {}, pos, parent, controllerMask)
	set_box(openingMaskLeft, -(maskSide + 5), 0, maskSide + 5, -(maskSide + 5))
	openingMaskLeft.init_rot = {rot}
			
	addStrokeArc(name.."_left", radius, 180, rot + 180, pos, parent, controllerMain, 0)

	local closingMaskLeft = closeMaskArea(level + 1, name.."_closingMaskLeft",
		openingMaskLeft.vertices, openingMaskLeft.indices, openingMaskLeft.init_pos, parent, controllerMask)
	openingMaskLeft.init_rot = {rot}
end

function addStrokeArc(name, radius, arc, rot, pos, parent, controllers, level)
	local obj = CreateElement "ceSMultiLine"

	local MaxSegments = 64
	
	local arc = arc or 360
	if arc > 360 then
		arc = 360
	end
		
	local count = math.ceil(arc / 360 * MaxSegments)
	local delta = math.rad(arc/count)
	
	setSymbolCommonProperties(obj, name, pos, parent, controllers)
	
	if rot ~= nil then
		obj.init_rot   = {rot}
	end
	
	setStrokeSymbolProperties(obj)
	
	local verts    = {}
	local inds     = {}

	for i = 1, count + 1 do
		verts[i] = {radius * math.sin(delta * (i - 1)), radius * math.cos(delta * (i - 1))}
	end
	
	for i = 1, count do
		inds[2*(i-1) + 1] = i - 1
		inds[2*(i-1) + 2] = i
	end

	obj.vertices         = verts              
	obj.indices          = inds

	setClipLevel(obj, level or 0)
	
	Add(obj)
	return obj
end

function addTexturedMesh(name, material, vertices, tex_params, pos, rot, parent, controllers)
	if material == nil then
		do return end
	end
	
	local texturedMexh		= CreateElement "ceTexPoly"
	setSymbolCommonProperties(texturedMexh, name, pos, parent, controllers, material)
	
	texturedMexh.indices		= default_box_indices
	texturedMexh.vertices		= vertices
	texturedMexh.tex_params 	= tex_params 
	
	if rot ~= nil then
		texturedMexh.init_rot   = {rot}
	end
	
	Add(texturedMexh)
	return texturedMexh
end

-- Shaped arrow contour. Is used for SPIN format left/right arrows, also for HUD GPWS/TAWS Recovery Cue
function addArrowContour(Name, arrLength, arrWidth, headLength, headWidth, pos, rot, parent, controllers, isBack, backDeep)
	if arrLength > headLength then
		if headWidth > arrWidth then
			local ArrName = "Arrow_"..Name

			local ArrPH = addPlaceholder(ArrName, pos, parent, controllers)
			addStrokeLine(ArrName.."_Body_Top", arrLength - headLength, {arrLength/2 , arrWidth/2}, 90, ArrName)
			addStrokeLine(ArrName.."_Body_Down", arrLength - headLength, {arrLength/2 , -arrWidth/2}, 90, ArrName)
			addStrokeLine(ArrName.."_Head_Back_Top", (headWidth - arrWidth) / 2, {-(arrLength/2 - headLength), arrWidth/2}, 0, ArrName)
			addStrokeLine(ArrName.."_Head_Back_Down", (headWidth - arrWidth) / 2, {-(arrLength/2 - headLength), -arrWidth/2}, 180, ArrName)
			local Hypotenuse = math.ceil(math.sqrt((headWidth / 2)^2 + headLength ^ 2))
			local Angle = math.ceil(math.deg(math.atan((headWidth / 2) / headLength)))

			addStrokeLine(ArrName.."_Head_Body_Top", Hypotenuse, {-arrLength/2, 0}, Angle - 90, ArrName)
			addStrokeLine(ArrName.."_Head_Body_Down", Hypotenuse, {-arrLength/2, 0}, -Angle - 90, ArrName)

			if isBack then
				local BackHypotenuse = math.ceil(math.sqrt((arrWidth / 2)^2 + backDeep ^ 2))
				local BackAngle = math.ceil(math.deg(math.atan((arrWidth / 2) / backDeep)))

				addStrokeLine(ArrName.."_Back_Body_Top", BackHypotenuse, {arrLength/2 - backDeep, 0}, BackAngle - 90, ArrName)
				addStrokeLine(ArrName.."_Back_Body_Down", BackHypotenuse, {arrLength/2 - backDeep, 0}, -BackAngle - 90, ArrName)
			else
				addStrokeLine(ArrName.."_Back", arrWidth, {arrLength/2 , -arrWidth/2}, 0, ArrName)
			end

			if rot ~= nil then
				ArrPH.init_rot = {rot}
			end
	
		end
	end
end

-- X over a display element - text, symbol, etc
function add_X_Over(name, width, height, pos, parent, controllers)
	local rootName = name.."_Root"
	local angle = math.floor(math.deg(math.atan(height / width)))
	local hypotenuse = math.floor(math.sqrt(height ^ 2 + width ^ 2))

	addPlaceholder(rootName, pos, parent, controllers)
	addStrokeLine(name.."_BottomLine", hypotenuse, {-width / 2, -height / 2}, -90 + angle, rootName)
	addStrokeLine(name.."_TopLine", hypotenuse, {-width / 2, height / 2}, -90 - angle, rootName)
end

-- The caret is pointed down by default
function addCaretByWidthHeight(name, width, height, pos, rot, parent, controllers)
	local elevCaretAngle   = math.atan((height/2)/width)
	local elevCaretLineLen = width / math.cos(elevCaretAngle)
	
	rot = rot or 0
	
	local caretPlaceholder = addPlaceholder(name.."_placeholder", pos, parent, controllers)
	addStrokeLine(name.."_line1", elevCaretLineLen, nil, rot + math.deg(elevCaretAngle), caretPlaceholder.name)
	addStrokeLine(name.."_line2", elevCaretLineLen, nil, rot - math.deg(elevCaretAngle), caretPlaceholder.name)
	
	return caretPlaceholder
end

-- The arrow is pointed up by default
function addArrow(name, length, pointerLen, pointerAngle, posShift, pos, rot, parent, controllers)
	rot = rot or 0
	
	local posAngle = math.rad(-rot)

	local arrowPlaceholder = addPlaceholder(name.."_placeholder", pos, parent, controllers)
	
	-- main line
	local arrowPos = {posShift * math.sin(posAngle), posShift * math.cos(posAngle)}
	addStrokeLine(name.."_mainLine", length, arrowPos, rot, arrowPlaceholder.name)
	
	-- pointer lines
	local pointerPosShift = posShift + length
	local pointerPos = {pointerPosShift * math.sin(posAngle), pointerPosShift * math.cos(posAngle)}
	
	local pointerAngleHalf = pointerAngle / 2
	addStrokeLine(name.."_pointerLine1", pointerLen, pointerPos, 180 + rot + pointerAngleHalf, arrowPlaceholder.name)
	addStrokeLine(name.."_pointerLine2", pointerLen, pointerPos, 180 + rot - pointerAngleHalf, arrowPlaceholder.name)
	
	return arrowPlaceholder
end

function addVarLenArrow(name, length, pointerLen, pointerAngle, posShift, pos, rot, parent, controllerRoot, controllerLen, controllerArrows)
	rot = rot or 0
	
	local posAngle = math.rad(-rot)

	local arrowPlaceholder = addPlaceholder(name.."_placeholder", pos, parent, controllerRoot)
	
	-- main line
	local arrowPos = {posShift * math.sin(posAngle), posShift * math.cos(posAngle)}
	addVarLenStrokeLine(name.."_mainLine", length, arrowPos, rot, arrowPlaceholder.name, controllerLen)
	
	-- pointer lines
	local pointerPosShift = posShift + length
	local pointerPos = {pointerPosShift * math.sin(posAngle), pointerPosShift * math.cos(posAngle)}
	
	local pointerAngleHalf = pointerAngle / 2
	local pointerLine = addPlaceholder(name.."_pointerLine_PH", nil, arrowPlaceholder.name, controllerArrows)
		addStrokeLine(name.."_pointerLine1", pointerLen, pointerPos, 180 + rot + pointerAngleHalf, pointerLine.name)
		addStrokeLine(name.."_pointerLine2", pointerLen, pointerPos, 180 + rot - pointerAngleHalf, pointerLine.name)
	
	return arrowPlaceholder
end

function addDot(name, radius, pos, parent, controllers)
	return addStrokeCircle(name, radius, pos, parent, controllers, nil, math.pi * 2 / 4)
end

-- TDC priority/sensor assignment indicator
function addMPD_TDC_diamond()
	local posX = 448
	local posY = 455
	local boxSide = 18
	
	local placeholder = addPlaceholder("TDC_diamond_placeholder", {posX, posY}, nil, {{"TDC_assignedDisplay"}})
	local box = addStrokeBox("TDC_diamond_box", boxSide, boxSide, "CenterCenter", nil, placeholder.name)
	-- make the diamond from the box 
	addStrokeCircle("TDC_diamond_box_DOT", 1, nil, box.name)
	box.init_rot = {45}
end
